<?php

if (!function_exists('perdas_normalize_header')) {
    function perdas_normalize_header(string $header): string {
        $h = trim($header);
        if (function_exists('mb_strtolower')) {
            $h = mb_strtolower($h, 'UTF-8');
        } else {
            $h = strtolower($h);
        }

        if (function_exists('iconv')) {
            $iconv = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $h);
            if ($iconv !== false) {
                $h = $iconv;
            }
        }

        $h = strtolower($h);
        $h = strtr($h, ['.' => ' ', '/' => ' ', '-' => ' ', '%' => ' ']);
        $h = preg_replace('/[^a-z0-9 ]+/', ' ', $h);
        $h = preg_replace('/\s+/', ' ', $h);
        return trim((string)$h);
    }
}

if (!function_exists('perdas_parse_int')) {
    function perdas_parse_int($v): int {
        if ($v === null || $v === '') return 0;
        if (is_int($v)) return $v;
        if (is_float($v)) return (int)round($v);

        $s = trim((string)$v);
        if ($s === '') return 0;
        $s = str_replace([' ', '.'], ['', ''], $s);
        $s = str_replace(',', '.', $s);
        return is_numeric($s) ? (int)round((float)$s) : 0;
    }
}

if (!function_exists('perdas_parse_decimal')) {
    function perdas_parse_decimal($v): ?float {
        if ($v === null || $v === '') return null;
        if (is_int($v) || is_float($v)) return (float)$v;

        $s = trim((string)$v);
        if ($s === '') return null;
        $s = str_replace([' ', '.'], ['', ''], $s);
        $s = str_replace(',', '.', $s);
        return is_numeric($s) ? (float)$s : null;
    }
}

if (!function_exists('perdas_parse_date')) {
    function perdas_parse_date($v): ?string {
        if ($v === null || $v === '') return null;

        if (is_int($v) || is_float($v)) {
            $excelDays = (float)$v;
            $unix = (int)round(($excelDays - 25569) * 86400);
            if ($unix > 0) {
                return gmdate('Y-m-d', $unix);
            }
        }

        $s = trim((string)$v);
        if ($s === '') return null;

        $ts = strtotime($s);
        if ($ts !== false) {
            return date('Y-m-d', $ts);
        }

        return null;
    }
}

if (!function_exists('perdas_header_map')) {
    function perdas_header_map(array $headerRow): array {
        $aliases = [
            'op trimb' => 'op_trimb',
            'op trimbox' => 'op_trimb',
            'cliente' => 'cliente',
            'pedido' => 'pedido',
            'valor unitario' => 'valor_unitario',
            'valor unit ario' => 'valor_unitario',
            'valor total previsto' => 'valor_total_previsto',
            'realizado' => 'realizado',
            'valor total realizado' => 'valor_total_realizado',
            'material' => 'material',
            'chapa reservada faturada' => 'chapa_reservada_faturada',
            'contador alimentacao' => 'contador_alimentacao',
            'contador alimentac ao' => 'contador_alimentacao',
            'conferencia' => 'conferencia',
            'confer encia' => 'conferencia',
            'dt producao' => 'dt_producao',
            'dt produc ao' => 'dt_producao',
            'perda setup' => 'perda_setup',
            'abaulada' => 'abaulada',
            'impressao nao conforme' => 'impressao_nao_conforme',
            'impress ao n ao conforme' => 'impressao_nao_conforme',
            'materia prima' => 'materia_prima',
            'mat eria prima' => 'materia_prima',
            'acabou cola' => 'acabou_cola',
            'acabou tinta' => 'acabou_tinta',
            'delaminacao' => 'delaminacao',
            'delaminac ao' => 'delaminacao',
            'fora de esquadro' => 'fora_de_esquadro',
            'rasgada' => 'rasgada',
            'enroscou' => 'enroscou',
            'outros' => 'outros',
            'perda total' => 'perda_total',
        ];

        $map = [];
        foreach ($headerRow as $idx => $label) {
            $k = perdas_normalize_header((string)$label);
            if (isset($aliases[$k])) {
                $map[$aliases[$k]] = $idx;
            }
        }
        return $map;
    }
}

if (!function_exists('perdas_transform_row')) {
    function perdas_transform_row(array $row, array $map): ?array {
        $get = static function (string $field) use ($row, $map) {
            if (!isset($map[$field])) return null;
            $idx = $map[$field];
            return $row[$idx] ?? null;
        };

        $op = trim((string)$get('op_trimb'));
        $cliente = trim((string)$get('cliente'));
        $dt = perdas_parse_date($get('dt_producao'));

        if ($op === '' || $cliente === '' || !$dt) {
            return null;
        }

        $item = [
            'dt_producao' => $dt,
            'op_trimb' => $op,
            'cliente' => $cliente,
            'pedido' => perdas_parse_int($get('pedido')),
            'valor_unitario' => perdas_parse_decimal($get('valor_unitario')),
            'valor_total_previsto' => perdas_parse_decimal($get('valor_total_previsto')),
            'realizado' => perdas_parse_int($get('realizado')),
            'valor_total_realizado' => perdas_parse_decimal($get('valor_total_realizado')),
            'material' => trim((string)$get('material')),
            'chapa_reservada_faturada' => perdas_parse_int($get('chapa_reservada_faturada')),
            'contador_alimentacao' => perdas_parse_int($get('contador_alimentacao')),
            'conferencia' => perdas_parse_int($get('conferencia')),
            'perda_setup' => perdas_parse_int($get('perda_setup')),
            'abaulada' => perdas_parse_int($get('abaulada')),
            'impressao_nao_conforme' => perdas_parse_int($get('impressao_nao_conforme')),
            'materia_prima' => perdas_parse_int($get('materia_prima')),
            'acabou_cola' => perdas_parse_int($get('acabou_cola')),
            'acabou_tinta' => perdas_parse_int($get('acabou_tinta')),
            'delaminacao' => perdas_parse_int($get('delaminacao')),
            'fora_de_esquadro' => perdas_parse_int($get('fora_de_esquadro')),
            'rasgada' => perdas_parse_int($get('rasgada')),
            'enroscou' => perdas_parse_int($get('enroscou')),
            'outros' => perdas_parse_int($get('outros')),
            'justificativa_outros' => '',
        ];

        $sumPerdas =
            $item['perda_setup'] +
            $item['abaulada'] +
            $item['impressao_nao_conforme'] +
            $item['materia_prima'] +
            $item['acabou_cola'] +
            $item['acabou_tinta'] +
            $item['delaminacao'] +
            $item['fora_de_esquadro'] +
            $item['rasgada'] +
            $item['enroscou'] +
            $item['outros'];

        $fromSheet = perdas_parse_int($get('perda_total'));
        $item['perda_total'] = max($sumPerdas, $fromSheet);

        $base = max(1, $item['chapa_reservada_faturada']);
        $item['perc_perda'] = round(($item['perda_total'] / $base) * 100, 2);
        $item['diferenca_alimentacao'] = $item['contador_alimentacao'] - $item['chapa_reservada_faturada'];

        return $item;
    }
}

if (!function_exists('perdas_import_from_xlsx')) {
    function perdas_import_from_xlsx(string $filePath, PDO $pdo, int $userId): array {
        if (!class_exists('Shuchkin\\SimpleXLSX')) {
            throw new RuntimeException('Dependencia de importacao ausente. Rode: composer install');
        }

        $xlsx = \Shuchkin\SimpleXLSX::parse($filePath);
        if (!$xlsx) {
            throw new RuntimeException('Nao foi possivel ler o arquivo XLSX.');
        }

        $sheetNames = $xlsx->sheetNames();
        $result = [
            'sheets' => $sheetNames,
            'inserted' => 0,
            'updated' => 0,
            'skipped' => 0,
            'processed' => 0,
        ];

        $stFindWithPedido = $pdo->prepare('SELECT id FROM bi_perdas WHERE dt_producao = ? AND op_trimb = ? AND cliente = ? AND pedido = ? LIMIT 1');
        $stFindNoPedido = $pdo->prepare('SELECT id FROM bi_perdas WHERE dt_producao = ? AND op_trimb = ? AND cliente = ? AND (pedido IS NULL OR pedido = 0) LIMIT 1');

        $sqlUpdate = "UPDATE bi_perdas SET
            valor_unitario=:valor_unitario,
            valor_total_previsto=:valor_total_previsto,
            realizado=:realizado,
            valor_total_realizado=:valor_total_realizado,
            material=:material,
            chapa_reservada_faturada=:chapa_reservada_faturada,
            contador_alimentacao=:contador_alimentacao,
            conferencia=:conferencia,
            perda_setup=:perda_setup,
            abaulada=:abaulada,
            impressao_nao_conforme=:impressao_nao_conforme,
            materia_prima=:materia_prima,
            acabou_cola=:acabou_cola,
            acabou_tinta=:acabou_tinta,
            delaminacao=:delaminacao,
            fora_de_esquadro=:fora_de_esquadro,
            rasgada=:rasgada,
            enroscou=:enroscou,
            outros=:outros,
            perda_total=:perda_total,
            perc_perda=:perc_perda,
            diferenca_alimentacao=:diferenca_alimentacao,
            justificativa_outros=:justificativa_outros,
            updated_at=NOW()
          WHERE id=:id";
        $stUpdate = $pdo->prepare($sqlUpdate);

        $sqlInsert = "INSERT INTO bi_perdas (
            dt_producao, op_trimb, cliente, pedido,
            valor_unitario, valor_total_previsto, realizado, valor_total_realizado,
            material, chapa_reservada_faturada, contador_alimentacao, conferencia,
            perda_setup, abaulada, impressao_nao_conforme, materia_prima, acabou_cola, acabou_tinta,
            delaminacao, fora_de_esquadro, rasgada, enroscou, outros, perda_total, perc_perda,
            diferenca_alimentacao, justificativa_outros, created_by, created_at, updated_at
          ) VALUES (
            :dt_producao, :op_trimb, :cliente, :pedido,
            :valor_unitario, :valor_total_previsto, :realizado, :valor_total_realizado,
            :material, :chapa_reservada_faturada, :contador_alimentacao, :conferencia,
            :perda_setup, :abaulada, :impressao_nao_conforme, :materia_prima, :acabou_cola, :acabou_tinta,
            :delaminacao, :fora_de_esquadro, :rasgada, :enroscou, :outros, :perda_total, :perc_perda,
            :diferenca_alimentacao, :justificativa_outros, :created_by, NOW(), NOW()
          )";
        $stInsert = $pdo->prepare($sqlInsert);

        $pdo->beginTransaction();
        try {
            foreach ($sheetNames as $sheetIndex => $sheetName) {
                $rows = $xlsx->rows($sheetIndex);
                if (!$rows || count($rows) < 2) continue;

                $headerMap = perdas_header_map($rows[0]);
                if (!isset($headerMap['op_trimb'], $headerMap['cliente'], $headerMap['dt_producao'])) {
                    continue;
                }

                for ($i = 1; $i < count($rows); $i++) {
                    $item = perdas_transform_row($rows[$i], $headerMap);
                    if (!$item) {
                        $result['skipped']++;
                        continue;
                    }

                    $result['processed']++;

                    $existingId = null;
                    if ($item['pedido'] > 0) {
                        $stFindWithPedido->execute([$item['dt_producao'], $item['op_trimb'], $item['cliente'], $item['pedido']]);
                        $existingId = $stFindWithPedido->fetchColumn();
                    } else {
                        $stFindNoPedido->execute([$item['dt_producao'], $item['op_trimb'], $item['cliente']]);
                        $existingId = $stFindNoPedido->fetchColumn();
                    }

                    if ($existingId) {
                        $stUpdate->execute([
                            'valor_unitario' => $item['valor_unitario'],
                            'valor_total_previsto' => $item['valor_total_previsto'],
                            'realizado' => $item['realizado'],
                            'valor_total_realizado' => $item['valor_total_realizado'],
                            'material' => $item['material'],
                            'chapa_reservada_faturada' => $item['chapa_reservada_faturada'],
                            'contador_alimentacao' => $item['contador_alimentacao'],
                            'conferencia' => $item['conferencia'],
                            'perda_setup' => $item['perda_setup'],
                            'abaulada' => $item['abaulada'],
                            'impressao_nao_conforme' => $item['impressao_nao_conforme'],
                            'materia_prima' => $item['materia_prima'],
                            'acabou_cola' => $item['acabou_cola'],
                            'acabou_tinta' => $item['acabou_tinta'],
                            'delaminacao' => $item['delaminacao'],
                            'fora_de_esquadro' => $item['fora_de_esquadro'],
                            'rasgada' => $item['rasgada'],
                            'enroscou' => $item['enroscou'],
                            'outros' => $item['outros'],
                            'perda_total' => $item['perda_total'],
                            'perc_perda' => $item['perc_perda'],
                            'diferenca_alimentacao' => $item['diferenca_alimentacao'],
                            'justificativa_outros' => $item['justificativa_outros'],
                            'id' => (int)$existingId,
                        ]);
                        $result['updated']++;
                    } else {
                        $stInsert->execute([
                            'dt_producao' => $item['dt_producao'],
                            'op_trimb' => $item['op_trimb'],
                            'cliente' => $item['cliente'],
                            'pedido' => $item['pedido'],
                            'valor_unitario' => $item['valor_unitario'],
                            'valor_total_previsto' => $item['valor_total_previsto'],
                            'realizado' => $item['realizado'],
                            'valor_total_realizado' => $item['valor_total_realizado'],
                            'material' => $item['material'],
                            'chapa_reservada_faturada' => $item['chapa_reservada_faturada'],
                            'contador_alimentacao' => $item['contador_alimentacao'],
                            'conferencia' => $item['conferencia'],
                            'perda_setup' => $item['perda_setup'],
                            'abaulada' => $item['abaulada'],
                            'impressao_nao_conforme' => $item['impressao_nao_conforme'],
                            'materia_prima' => $item['materia_prima'],
                            'acabou_cola' => $item['acabou_cola'],
                            'acabou_tinta' => $item['acabou_tinta'],
                            'delaminacao' => $item['delaminacao'],
                            'fora_de_esquadro' => $item['fora_de_esquadro'],
                            'rasgada' => $item['rasgada'],
                            'enroscou' => $item['enroscou'],
                            'outros' => $item['outros'],
                            'perda_total' => $item['perda_total'],
                            'perc_perda' => $item['perc_perda'],
                            'diferenca_alimentacao' => $item['diferenca_alimentacao'],
                            'justificativa_outros' => $item['justificativa_outros'],
                            'created_by' => $userId,
                        ]);
                        $result['inserted']++;
                    }
                }
            }

            $pdo->commit();
        } catch (Throwable $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            throw $e;
        }

        return $result;
    }
}

