<?php
require_auth();
require __DIR__ . '/_layout_top.php';

function valid_date_dash($value) {
  return is_string($value) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $value) && strtotime($value) !== false;
}

function period_from_request_dash($prefix = '') {
  $mode = isset($_GET[$prefix . 'mode']) ? (string)$_GET[$prefix . 'mode'] : 'month';
  if (!in_array($mode, ['day', 'week', 'month', 'custom'], true)) {
    $mode = 'month';
  }

  if ($mode === 'day') {
    $day = isset($_GET[$prefix . 'day']) ? (string)$_GET[$prefix . 'day'] : date('Y-m-d');
    if (!valid_date_dash($day)) $day = date('Y-m-d');
    return ['mode' => 'day', 'ini' => $day, 'fim' => $day, 'label' => date('d/m/Y', strtotime($day))];
  }

  if ($mode === 'week') {
    $week = isset($_GET[$prefix . 'week']) ? (string)$_GET[$prefix . 'week'] : date('o-\\WW');
    if (!preg_match('/^(\d{4})-W(\d{2})$/', $week, $m)) {
      $week = date('o-\\WW');
      preg_match('/^(\d{4})-W(\d{2})$/', $week, $m);
    }
    $dt = new DateTime();
    $dt->setISODate((int)$m[1], (int)$m[2]);
    $ini = $dt->format('Y-m-d');
    $dt->modify('+6 days');
    $fim = $dt->format('Y-m-d');
    return ['mode' => 'week', 'ini' => $ini, 'fim' => $fim, 'label' => date('d/m', strtotime($ini)) . ' a ' . date('d/m', strtotime($fim))];
  }

  if ($mode === 'custom') {
    $ini = isset($_GET[$prefix . 'dt_ini']) ? (string)$_GET[$prefix . 'dt_ini'] : date('Y-m-01');
    $fim = isset($_GET[$prefix . 'dt_fim']) ? (string)$_GET[$prefix . 'dt_fim'] : date('Y-m-d');
    if (!valid_date_dash($ini) || !valid_date_dash($fim)) {
      $ini = date('Y-m-01');
      $fim = date('Y-m-d');
    }
    if ($fim < $ini) {
      $tmp = $ini;
      $ini = $fim;
      $fim = $tmp;
    }
    return ['mode' => 'custom', 'ini' => $ini, 'fim' => $fim, 'label' => date('d/m/Y', strtotime($ini)) . ' a ' . date('d/m/Y', strtotime($fim))];
  }

  $mes = isset($_GET[$prefix . 'mes']) ? (string)$_GET[$prefix . 'mes'] : date('Y-m');
  if (!preg_match('/^\d{4}-\d{2}$/', $mes)) {
    $mes = date('Y-m');
  }
  $ini = $mes . '-01';
  $fim = date('Y-m-t', strtotime($ini));
  return ['mode' => 'month', 'ini' => $ini, 'fim' => $fim, 'label' => date('m/Y', strtotime($ini))];
}

function previous_period_dash($ini, $fim) {
  $days = (int)floor((strtotime($fim) - strtotime($ini)) / 86400) + 1;
  $prevFim = date('Y-m-d', strtotime($ini . ' -1 day'));
  $prevIni = date('Y-m-d', strtotime($prevFim . ' -' . ($days - 1) . ' day'));
  return ['mode' => 'custom', 'ini' => $prevIni, 'fim' => $prevFim, 'label' => date('d/m/Y', strtotime($prevIni)) . ' a ' . date('d/m/Y', strtotime($prevFim))];
}

$period = period_from_request_dash('');
$compareEnabled = isset($_GET['compare']) && (string)$_GET['compare'] === '1';
$comparePeriod = null;
if ($compareEnabled) {
  $hasCompareInputs = isset($_GET['compare_mode']) || isset($_GET['compare_mes']) || isset($_GET['compare_day']) || isset($_GET['compare_week']) || isset($_GET['compare_dt_ini']) || isset($_GET['compare_dt_fim']);
  $comparePeriod = $hasCompareInputs ? period_from_request_dash('compare_') : previous_period_dash($period['ini'], $period['fim']);
}

$st = db()->prepare("SELECT
  dt_producao, op_trimb, cliente, pedido, valor_unitario, valor_total_previsto,
  realizado, valor_total_realizado, material, chapa_reservada_faturada,
  contador_alimentacao, conferencia, perda_total
FROM bi_perdas
WHERE dt_producao BETWEEN ? AND ?
ORDER BY dt_producao DESC, id DESC
LIMIT 200");
$st->execute([$period['ini'], $period['fim']]);
$gridRows = $st->fetchAll();
?>

<section class="glass-card p-5 md:p-6" id="sec-kpis">
  <div class="flex flex-wrap items-center justify-between gap-3">
    <div>
      <p class="text-xs uppercase tracking-[0.15em] text-mnc-muted">Visao Executiva</p>
      <h1 class="mt-1 text-2xl font-extrabold">Controle de Perdas e Saldos</h1>
      <p class="mt-1 text-sm text-mnc-muted">Periodo atual: <strong><?= e($period['label']) ?></strong></p>
      <?php if ($compareEnabled && $comparePeriod): ?>
        <p class="mt-1 text-sm text-mnc-muted">Comparando com: <strong><?= e($comparePeriod['label']) ?></strong></p>
      <?php endif; ?>
    </div>
  </div>

  <form method="get" action="index.php" class="mt-4 grid gap-3 xl:grid-cols-2">
    <input type="hidden" name="p" value="dashboard">
    <input type="hidden" name="m" value="<?= e(current_module_key()) ?>">

    <fieldset class="rounded-2xl border border-slate-200 bg-white p-4">
      <legend class="px-2 text-xs font-bold uppercase tracking-[0.14em] text-mnc-muted">Periodo A (Base)</legend>
      <div class="grid gap-2 md:grid-cols-2">
        <div>
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Tipo</label>
          <select name="mode" id="mode" class="input-dark">
            <option value="month" <?= $period['mode'] === 'month' ? 'selected' : '' ?>>Mes</option>
            <option value="week" <?= $period['mode'] === 'week' ? 'selected' : '' ?>>Semana</option>
            <option value="day" <?= $period['mode'] === 'day' ? 'selected' : '' ?>>Dia</option>
            <option value="custom" <?= $period['mode'] === 'custom' ? 'selected' : '' ?>>Periodo</option>
          </select>
        </div>
        <div class="period-field period-month">
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Mes</label>
          <input type="month" name="mes" class="input-dark" value="<?= e($_GET['mes'] ?? date('Y-m')) ?>">
        </div>
        <div class="period-field period-week">
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Semana</label>
          <input type="week" name="week" class="input-dark" value="<?= e($_GET['week'] ?? date('o-\\WW')) ?>">
        </div>
        <div class="period-field period-day">
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Dia</label>
          <input type="date" name="day" class="input-dark" value="<?= e($_GET['day'] ?? date('Y-m-d')) ?>">
        </div>
        <div class="period-field period-custom">
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Data Inicial</label>
          <input type="date" name="dt_ini" class="input-dark" value="<?= e($_GET['dt_ini'] ?? $period['ini']) ?>">
        </div>
        <div class="period-field period-custom">
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Data Final</label>
          <input type="date" name="dt_fim" class="input-dark" value="<?= e($_GET['dt_fim'] ?? $period['fim']) ?>">
        </div>
      </div>
    </fieldset>

    <fieldset class="rounded-2xl border border-slate-200 bg-white p-4">
      <legend class="px-2 text-xs font-bold uppercase tracking-[0.14em] text-mnc-muted">Periodo B (Comparacao)</legend>
      <label class="mb-2 flex items-center gap-2 text-sm font-semibold text-slate-700">
        <input type="checkbox" id="compare" name="compare" value="1" <?= $compareEnabled ? 'checked' : '' ?>>
        Habilitar comparacao periodo A vs periodo B
      </label>

      <div id="compareWrap" class="grid gap-2 md:grid-cols-2 <?= $compareEnabled ? '' : 'hidden' ?>">
        <div>
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Tipo</label>
          <select name="compare_mode" id="compare_mode" class="input-dark">
            <?php $cm = $_GET['compare_mode'] ?? ($comparePeriod['mode'] ?? 'custom'); ?>
            <option value="month" <?= $cm === 'month' ? 'selected' : '' ?>>Mes</option>
            <option value="week" <?= $cm === 'week' ? 'selected' : '' ?>>Semana</option>
            <option value="day" <?= $cm === 'day' ? 'selected' : '' ?>>Dia</option>
            <option value="custom" <?= $cm === 'custom' ? 'selected' : '' ?>>Periodo</option>
          </select>
        </div>
        <div class="compare-field compare-month">
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Mes</label>
          <input type="month" name="compare_mes" class="input-dark" value="<?= e($_GET['compare_mes'] ?? date('Y-m', strtotime(($comparePeriod['ini'] ?? date('Y-m-d'))))) ?>">
        </div>
        <div class="compare-field compare-week">
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Semana</label>
          <input type="week" name="compare_week" class="input-dark" value="<?= e($_GET['compare_week'] ?? date('o-\\WW', strtotime(($comparePeriod['ini'] ?? date('Y-m-d'))))) ?>">
        </div>
        <div class="compare-field compare-day">
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Dia</label>
          <input type="date" name="compare_day" class="input-dark" value="<?= e($_GET['compare_day'] ?? ($comparePeriod['ini'] ?? date('Y-m-d'))) ?>">
        </div>
        <div class="compare-field compare-custom">
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Data Inicial</label>
          <input type="date" name="compare_dt_ini" class="input-dark" value="<?= e($_GET['compare_dt_ini'] ?? ($comparePeriod['ini'] ?? date('Y-m-01'))) ?>">
        </div>
        <div class="compare-field compare-custom">
          <label class="mb-1 block text-xs font-semibold uppercase tracking-[0.12em] text-mnc-muted">Data Final</label>
          <input type="date" name="compare_dt_fim" class="input-dark" value="<?= e($_GET['compare_dt_fim'] ?? ($comparePeriod['fim'] ?? date('Y-m-d'))) ?>">
        </div>
      </div>
    </fieldset>

    <div class="xl:col-span-2 flex flex-wrap justify-end gap-2">
      <button class="btn-primary">Aplicar Analise</button>
      <button type="button" class="btn-muted" onclick="window.location.href='<?= e(route_url('dashboard')) ?>'">Limpar</button>
    </div>
  </form>

  <div class="mt-3 flex flex-wrap gap-2">
    <button type="button" class="btn-primary" onclick="window.location.href='<?= e(route_url('perdas_form')) ?>'">Novo Registro</button>
    <button type="button" class="btn-muted" onclick="window.location.href='<?= e(route_url('perdas')) ?>'">Ver Registros</button>
    <button type="button" class="btn-muted" onclick="window.location.href='<?= e(route_url('perdas_import')) ?>'">Importar XLSX</button>
  </div>

  <div class="mt-4 flex flex-wrap gap-2 text-sm">
    <button type="button" class="nav-pill" onclick="document.getElementById('sec-kpis').scrollIntoView({behavior:'smooth'})">KPIs</button>
    <button type="button" class="nav-pill" onclick="document.getElementById('sec-compare').scrollIntoView({behavior:'smooth'})">Comparativo</button>
    <button type="button" class="nav-pill" onclick="document.getElementById('sec-charts').scrollIntoView({behavior:'smooth'})">Graficos</button>
    <button type="button" class="nav-pill" onclick="document.getElementById('sec-rankings').scrollIntoView({behavior:'smooth'})">Rankings</button>
    <button type="button" class="nav-pill" onclick="document.getElementById('sec-planilha').scrollIntoView({behavior:'smooth'})">Tabela</button>
  </div>

  <div class="mt-4 grid gap-3 sm:grid-cols-2 xl:grid-cols-5">
    <div class="rounded-2xl border border-slate-200 bg-white p-4"><div class="kpi-value" id="kpi_perda_total">-</div><div class="kpi-label">Perda Total (un.)</div></div>
    <div class="rounded-2xl border border-slate-200 bg-white p-4"><div class="kpi-value" id="kpi_perc_media">-</div><div class="kpi-label">% Perda Media</div></div>
    <div class="rounded-2xl border border-slate-200 bg-white p-4"><div class="kpi-value" id="kpi_saldo_unidades">-</div><div class="kpi-label">Saldo Unidades</div></div>
    <div class="rounded-2xl border border-slate-200 bg-white p-4"><div class="kpi-value" id="kpi_valor_previsto">-</div><div class="kpi-label">Valor Previsto</div></div>
    <div class="rounded-2xl border border-slate-200 bg-white p-4"><div class="kpi-value" id="kpi_saldo_financeiro">-</div><div class="kpi-label">Saldo Financeiro</div></div>
  </div>

  <div class="mt-3 grid gap-3 sm:grid-cols-2 xl:grid-cols-4">
    <div class="rounded-xl border border-slate-200 bg-white p-3"><div class="text-xs uppercase tracking-[0.12em] text-mnc-muted">Registros</div><div class="mt-1 text-xl font-extrabold" id="kpi_registros">-</div></div>
    <div class="rounded-xl border border-slate-200 bg-white p-3"><div class="text-xs uppercase tracking-[0.12em] text-mnc-muted">Pedido Total</div><div class="mt-1 text-xl font-extrabold" id="kpi_pedido_total">-</div></div>
    <div class="rounded-xl border border-slate-200 bg-white p-3"><div class="text-xs uppercase tracking-[0.12em] text-mnc-muted">Realizado Total</div><div class="mt-1 text-xl font-extrabold" id="kpi_realizado_total">-</div></div>
    <div class="rounded-xl border border-slate-200 bg-white p-3"><div class="text-xs uppercase tracking-[0.12em] text-mnc-muted">Valor Realizado</div><div class="mt-1 text-xl font-extrabold" id="kpi_valor_realizado">-</div></div>
  </div>
</section>

<section class="glass-card mt-4 p-5 md:p-6 <?= $compareEnabled ? '' : 'hidden' ?>" id="sec-compare">
  <div class="flex items-center justify-between gap-2">
    <h3 class="text-lg font-bold">Resultado Periodo A vs Periodo B</h3>
    <span class="text-xs uppercase tracking-[0.12em] text-mnc-muted">Delta = A - B</span>
  </div>
  <div class="mt-4 grid gap-3 sm:grid-cols-2 xl:grid-cols-4">
    <div class="rounded-xl border border-slate-200 bg-white p-4">
      <div class="text-xs uppercase tracking-[0.12em] text-mnc-muted">Perda Total</div>
      <div class="mt-1 text-lg font-extrabold" id="cmp_perda_base">-</div>
      <div class="text-xs text-slate-500" id="cmp_perda_ref">-</div>
      <div class="mt-1 text-sm font-bold" id="cmp_perda_delta">-</div>
    </div>
    <div class="rounded-xl border border-slate-200 bg-white p-4">
      <div class="text-xs uppercase tracking-[0.12em] text-mnc-muted">Saldo Unidades</div>
      <div class="mt-1 text-lg font-extrabold" id="cmp_saldo_base">-</div>
      <div class="text-xs text-slate-500" id="cmp_saldo_ref">-</div>
      <div class="mt-1 text-sm font-bold" id="cmp_saldo_delta">-</div>
    </div>
    <div class="rounded-xl border border-slate-200 bg-white p-4">
      <div class="text-xs uppercase tracking-[0.12em] text-mnc-muted">Valor Realizado</div>
      <div class="mt-1 text-lg font-extrabold" id="cmp_real_base">-</div>
      <div class="text-xs text-slate-500" id="cmp_real_ref">-</div>
      <div class="mt-1 text-sm font-bold" id="cmp_real_delta">-</div>
    </div>
    <div class="rounded-xl border border-slate-200 bg-white p-4">
      <div class="text-xs uppercase tracking-[0.12em] text-mnc-muted">Saldo Financeiro</div>
      <div class="mt-1 text-lg font-extrabold" id="cmp_fin_base">-</div>
      <div class="text-xs text-slate-500" id="cmp_fin_ref">-</div>
      <div class="mt-1 text-sm font-bold" id="cmp_fin_delta">-</div>
    </div>
  </div>
</section>

<section class="mt-4 grid gap-4 xl:grid-cols-2" id="sec-charts">
  <article class="glass-card p-5 md:p-6"><div class="flex items-center justify-between"><h3 class="text-lg font-bold">Perda por Motivo</h3><span class="text-xs uppercase tracking-[0.15em] text-mnc-muted">Periodo A</span></div><div class="mt-4"><canvas id="chartMotivos" height="120"></canvas></div></article>
  <article class="glass-card p-5 md:p-6"><div class="flex items-center justify-between"><h3 class="text-lg font-bold">Materiais com Maior Perda</h3><span class="text-xs uppercase tracking-[0.15em] text-mnc-muted">Top 8</span></div><div class="mt-4"><canvas id="chartMateriais" height="120"></canvas></div></article>
  <article class="glass-card p-5 md:p-6"><div class="flex items-center justify-between"><h3 class="text-lg font-bold">Top Clientes por Perda</h3><span class="text-xs uppercase tracking-[0.15em] text-mnc-muted">Top 8</span></div><div class="mt-4"><canvas id="chartClientes" height="150"></canvas></div></article>
  <article class="glass-card p-5 md:p-6"><div class="flex items-center justify-between"><h3 class="text-lg font-bold">Serie Diaria</h3><span class="text-xs uppercase tracking-[0.15em] text-mnc-muted">Perda x Saldo</span></div><div class="mt-4"><canvas id="chartSerie" height="150"></canvas></div></article>
</section>

<section class="mt-4 grid gap-4 xl:grid-cols-3" id="sec-rankings">
  <article class="glass-card p-5 md:p-6 xl:col-span-2">
    <div class="flex items-center justify-between gap-2"><h3 class="text-lg font-bold">Clientes com Maior Diferenca Pedido x Realizado</h3><span class="text-xs uppercase tracking-[0.15em] text-mnc-muted">Ordenado por modulo</span></div>
    <div class="mt-4 overflow-x-auto dt-grid-wrap"><table class="table-dark min-w-[760px]" id="dtClienteDiff"><thead><tr><th>Cliente</th><th class="text-right">Pedido</th><th class="text-right">Realizado</th><th class="text-right">Diferenca</th></tr></thead><tbody id="tblClienteDiff"></tbody></table></div>
  </article>
  <article class="glass-card p-5 md:p-6">
    <h3 class="text-lg font-bold">Materiais Criticos</h3>
    <p class="mt-1 text-sm text-mnc-muted">Perda e diferenca de volume por material.</p>
    <div class="mt-3 overflow-x-auto dt-grid-wrap"><table class="table-dark min-w-[420px]" id="dtMaterialDiff"><thead><tr><th>Material</th><th class="text-right">Perda</th><th class="text-right">Dif.</th></tr></thead><tbody id="tblMaterialDiff"></tbody></table></div>
  </article>
  <article class="glass-card p-5 md:p-6 xl:col-span-3">
    <div class="flex items-center justify-between gap-2"><h3 class="text-lg font-bold">Pedidos/OP com Maior Diferenca</h3><span class="text-xs uppercase tracking-[0.15em] text-mnc-muted">Top 10 por OP + Cliente</span></div>
    <div class="mt-4 overflow-x-auto dt-grid-wrap"><table class="table-dark min-w-[980px]" id="dtPedidoDiff"><thead><tr><th>OP</th><th>Cliente</th><th class="text-right">Pedido</th><th class="text-right">Realizado</th><th class="text-right">Diferenca</th></tr></thead><tbody id="tblPedidoDiff"></tbody></table></div>
  </article>
</section>

<section class="glass-card mt-4 p-5 md:p-6" id="sec-planilha">
  <div class="flex flex-wrap items-center justify-between gap-2"><h3 class="text-lg font-bold">Visao Operacional (Periodo A)</h3><span class="text-xs uppercase tracking-[0.15em] text-mnc-muted"><?= e($period['label']) ?> - ate 200 linhas</span></div>

  <div class="mt-3 grid gap-2 md:grid-cols-4">
    <input id="filtroCliente" class="input-dark" placeholder="Filtrar cliente...">
    <input id="filtroMaterial" class="input-dark" placeholder="Filtrar material...">
    <input id="filtroOp" class="input-dark" placeholder="Filtrar OP...">
    <select id="filtroDiferenca" class="input-dark"><option value="">Todos os saldos</option><option value="positivo">Somente diferenca positiva</option><option value="negativo">Somente diferenca negativa</option><option value="zero">Somente diferenca zero</option></select>
  </div>

  <div class="mt-4 overflow-x-auto table-wrap-sticky">
    <table class="table-dark table-sticky min-w-[1780px]" id="dtPlanilha">
      <thead><tr><th>OP TRIMBOX</th><th>CLIENTE</th><th>PEDIDO</th><th>REALIZADO</th><th>DIFERENCA</th><th>VALOR UNIT.</th><th>VALOR TOTAL PREVISTO</th><th>VALOR TOTAL REALIZADO</th><th>MATERIAL</th><th>CHAPA RESERVADA FATURADA</th><th>CONTADOR ALIMENTACAO</th><th>CONFERENCIA</th><th>DT. PRODUCAO</th><th>PERDA TOTAL</th></tr></thead>
      <tbody>
      <?php foreach ($gridRows as $r): ?>
        <?php $dif = (int)$r['realizado'] - (int)$r['pedido']; ?>
        <tr data-cliente="<?= e(strtolower((string)$r['cliente'])) ?>" data-material="<?= e(strtolower((string)$r['material'])) ?>" data-op="<?= e(strtolower((string)$r['op_trimb'])) ?>" data-dif="<?= e($dif) ?>">
          <td><?= e($r['op_trimb']) ?></td><td><?= e($r['cliente']) ?></td><td class="text-right"><?= e($r['pedido']) ?></td><td class="text-right"><?= e($r['realizado']) ?></td>
          <td class="text-right font-bold <?= $dif > 0 ? 'text-emerald-700' : ($dif < 0 ? 'text-red-700' : 'text-slate-500') ?>"><?= e($dif) ?></td>
          <td class="text-right"><?= e(num($r['valor_unitario'], 2)) ?></td><td class="text-right"><?= e(money($r['valor_total_previsto'])) ?></td><td class="text-right"><?= e(money($r['valor_total_realizado'])) ?></td>
          <td><?= e($r['material']) ?></td><td class="text-right"><?= e($r['chapa_reservada_faturada']) ?></td><td class="text-right"><?= e($r['contador_alimentacao']) ?></td><td class="text-right"><?= e($r['conferencia']) ?></td>
          <td><?= e(date('d/m/Y', strtotime($r['dt_producao']))) ?></td><td class="text-right font-bold text-cyan-700"><?= e($r['perda_total']) ?></td>
        </tr>
      <?php endforeach; ?>
      <?php if (!$gridRows): ?><tr><td colspan="14" class="py-6 text-center text-sm text-mnc-muted">Sem dados para o periodo.</td></tr><?php endif; ?>
      </tbody>
    </table>
  </div>
</section>

<script>
(function () {
  function togglePeriodFields(selectId, fieldPrefix) {
    const mode = document.getElementById(selectId).value;
    document.querySelectorAll('.' + fieldPrefix + '-field').forEach(function (el) {
      el.style.display = 'none';
    });
    document.querySelectorAll('.' + fieldPrefix + '-' + mode).forEach(function (el) {
      el.style.display = '';
    });
  }

  const modeEl = document.getElementById('mode');
  const compareModeEl = document.getElementById('compare_mode');
  const compareCheck = document.getElementById('compare');
  const compareWrap = document.getElementById('compareWrap');

  togglePeriodFields('mode', 'period');
  togglePeriodFields('compare_mode', 'compare');

  modeEl.addEventListener('change', function () { togglePeriodFields('mode', 'period'); });
  compareModeEl.addEventListener('change', function () { togglePeriodFields('compare_mode', 'compare'); });
  compareCheck.addEventListener('change', function () {
    compareWrap.classList.toggle('hidden', !compareCheck.checked);
  });

  const params = new URLSearchParams(window.location.search);
  params.set('p', 'api_kpis');
  if (!params.get('m')) params.set('m', <?= json_encode(current_module_key()) ?>);
  const endpoint = 'index.php?' + params.toString();

  function fmtNum(v, dec) {
    return Number(v || 0).toLocaleString('pt-BR', { minimumFractionDigits: dec || 0, maximumFractionDigits: dec || 0 });
  }

  function fmtMoney(v) {
    return Number(v || 0).toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });
  }

  function signClass(v) {
    if (v > 0) return 'text-emerald-700';
    if (v < 0) return 'text-red-700';
    return 'text-slate-500';
  }

  function fmtDelta(delta, pct, money) {
    const signal = delta > 0 ? '+' : '';
    const base = money ? fmtMoney(delta) : fmtNum(delta, 2);
    const pctTxt = pct === null || pct === undefined ? 'n/a' : (pct > 0 ? '+' : '') + fmtNum(pct, 2) + '%';
    return signal + base + ' (' + pctTxt + ')';
  }

  function fillTable(id, rows, renderRow, emptyCols) {
    const el = document.getElementById(id);
    if (!el) return;
    if (!rows || !rows.length) {
      el.innerHTML = '<tr><td colspan="' + (emptyCols || 5) + '" class="py-5 text-center text-sm text-slate-500">Sem dados no periodo.</td></tr>';
      return;
    }
    el.innerHTML = rows.map(renderRow).join('');
  }

  const dtRefs = {};
  let dtPlanilha = null;
  let planilhaDiffFilter = '';

  function dtLanguage() {
    return {
      search: 'Buscar:',
      lengthMenu: 'Mostrar _MENU_',
      info: 'Mostrando _START_ a _END_ de _TOTAL_',
      infoEmpty: 'Sem registros',
      emptyTable: 'Sem dados no periodo',
      paginate: { first: 'Primeiro', previous: 'Anterior', next: 'Proximo', last: 'Ultimo' }
    };
  }

  function initSimpleDataTable(id, opts) {
    if (!window.DataTable) return null;
    if (dtRefs[id]) {
      dtRefs[id].destroy();
      dtRefs[id] = null;
    }
    const tableEl = document.getElementById(id);
    if (!tableEl) return null;
    dtRefs[id] = new DataTable('#' + id, Object.assign({
      pageLength: 8,
      lengthMenu: [8, 16, 32, 64],
      language: dtLanguage()
    }, opts || {}));
    return dtRefs[id];
  }

  fetch(endpoint, { headers: { 'Accept': 'application/json' } })
    .then(function (res) { if (!res.ok) throw new Error('HTTP ' + res.status); return res.json(); })
    .then(function (data) {
      const k = data.kpis || {};
      document.getElementById('kpi_perda_total').textContent = fmtNum(k.perda_total, 0);
      document.getElementById('kpi_perc_media').textContent = fmtNum(k.perc_media, 2) + '%';
      document.getElementById('kpi_saldo_unidades').textContent = fmtNum(k.saldo_unidades, 0) + ' (' + fmtNum(k.saldo_perc, 2) + '%)';
      document.getElementById('kpi_valor_previsto').textContent = k.valor_previsto_fmt || 'R$ 0,00';
      document.getElementById('kpi_valor_realizado').textContent = k.valor_realizado_fmt || 'R$ 0,00';
      document.getElementById('kpi_saldo_financeiro').textContent = k.saldo_financeiro_fmt || 'R$ 0,00';
      document.getElementById('kpi_saldo_financeiro').classList.add(signClass(Number(k.saldo_financeiro || 0)));
      document.getElementById('kpi_registros').textContent = fmtNum(k.registros, 0);
      document.getElementById('kpi_pedido_total').textContent = fmtNum(k.pedido_total, 0);
      document.getElementById('kpi_realizado_total').textContent = fmtNum(k.realizado_total, 0);

      if (data.compare_enabled && data.kpis_compare && data.kpis_diff) {
        const kb = data.kpis_compare;
        const d = data.kpis_diff;
        document.getElementById('cmp_perda_base').textContent = fmtNum(k.perda_total, 0);
        document.getElementById('cmp_perda_ref').textContent = 'Ref: ' + fmtNum(kb.perda_total, 0);
        document.getElementById('cmp_perda_delta').textContent = fmtDelta(d.perda_total.delta, d.perda_total.pct, false);
        document.getElementById('cmp_perda_delta').classList.add(signClass(d.perda_total.delta));

        document.getElementById('cmp_saldo_base').textContent = fmtNum(k.saldo_unidades, 0);
        document.getElementById('cmp_saldo_ref').textContent = 'Ref: ' + fmtNum(kb.saldo_unidades, 0);
        document.getElementById('cmp_saldo_delta').textContent = fmtDelta(d.saldo_unidades.delta, d.saldo_unidades.pct, false);
        document.getElementById('cmp_saldo_delta').classList.add(signClass(d.saldo_unidades.delta));

        document.getElementById('cmp_real_base').textContent = fmtMoney(k.valor_realizado);
        document.getElementById('cmp_real_ref').textContent = 'Ref: ' + fmtMoney(kb.valor_realizado);
        document.getElementById('cmp_real_delta').textContent = fmtDelta(d.valor_realizado.delta, d.valor_realizado.pct, true);
        document.getElementById('cmp_real_delta').classList.add(signClass(d.valor_realizado.delta));

        document.getElementById('cmp_fin_base').textContent = fmtMoney(k.saldo_financeiro);
        document.getElementById('cmp_fin_ref').textContent = 'Ref: ' + fmtMoney(kb.saldo_financeiro);
        document.getElementById('cmp_fin_delta').textContent = fmtDelta(d.saldo_financeiro.delta, d.saldo_financeiro.pct, true);
        document.getElementById('cmp_fin_delta').classList.add(signClass(d.saldo_financeiro.delta));
      }

      const motivos = data.motivos || { labels: [], values: [] };
      new Chart(document.getElementById('chartMotivos'), { type: 'bar', data: { labels: motivos.labels, datasets: [{ label: 'Unidades', data: motivos.values, backgroundColor: 'rgba(40,199,246,0.75)', borderRadius: 8 }] }, options: { plugins: { legend: { display: false } }, scales: { x: { ticks: { color: '#475569' }, grid: { color: 'rgba(15,23,42,0.10)' } }, y: { ticks: { color: '#475569' }, grid: { color: 'rgba(15,23,42,0.10)' } } } } });

      const clientes = data.clientes || { labels: [], values: [] };
      new Chart(document.getElementById('chartClientes'), { type: 'doughnut', data: { labels: clientes.labels, datasets: [{ data: clientes.values, backgroundColor: ['#28c7f6','#2ce5a7','#93c5fd','#38bdf8','#22d3ee','#67e8f9','#bef264','#60a5fa'] }] }, options: { plugins: { legend: { position: 'bottom', labels: { color: '#475569', boxWidth: 12 } } } } });

      const material = data.material || { labels: [], values: [], rows: [] };
      new Chart(document.getElementById('chartMateriais'), { type: 'bar', data: { labels: material.labels, datasets: [{ label: 'Perda', data: material.values, backgroundColor: 'rgba(16,185,129,0.70)', borderRadius: 8 }] }, options: { indexAxis: 'y', plugins: { legend: { display: false } }, scales: { x: { ticks: { color: '#475569' }, grid: { color: 'rgba(15,23,42,0.10)' } }, y: { ticks: { color: '#475569' }, grid: { color: 'rgba(15,23,42,0.06)' } } } } });

      const serie = data.serie_diaria || { labels: [], perda: [], saldo: [] };
      new Chart(document.getElementById('chartSerie'), { type: 'line', data: { labels: serie.labels, datasets: [{ label: 'Perda', data: serie.perda, borderColor: '#06b6d4', backgroundColor: 'rgba(6,182,212,0.18)', tension: 0.3, fill: true }, { label: 'Saldo Unidades', data: serie.saldo, borderColor: '#f97316', backgroundColor: 'rgba(249,115,22,0.10)', tension: 0.3, fill: false }] }, options: { plugins: { legend: { labels: { color: '#475569' } } }, scales: { x: { ticks: { color: '#475569' }, grid: { color: 'rgba(15,23,42,0.08)' } }, y: { ticks: { color: '#475569' }, grid: { color: 'rgba(15,23,42,0.08)' } } } } });

      fillTable('tblClienteDiff', data.cliente_diferenca || [], function (r) { const dif = Number(r.diferenca || 0); return '<tr><td>' + (r.cliente || '(Sem cliente)') + '</td><td class="text-right">' + fmtNum(r.pedido_total, 0) + '</td><td class="text-right">' + fmtNum(r.realizado_total, 0) + '</td><td class="text-right font-bold ' + signClass(dif) + '">' + fmtNum(dif, 0) + '</td></tr>'; }, 4);
      fillTable('tblPedidoDiff', data.pedido_diferenca || [], function (r) { const dif = Number(r.diferenca || 0); return '<tr><td>' + (r.op_trimb || '') + '</td><td>' + (r.cliente || '(Sem cliente)') + '</td><td class="text-right">' + fmtNum(r.pedido_total, 0) + '</td><td class="text-right">' + fmtNum(r.realizado_total, 0) + '</td><td class="text-right font-bold ' + signClass(dif) + '">' + fmtNum(dif, 0) + '</td></tr>'; }, 5);
      fillTable('tblMaterialDiff', material.rows || [], function (r) { const dif = Number(r.diferenca || 0); return '<tr><td>' + (r.material || '(Sem material)') + '</td><td class="text-right">' + fmtNum(r.perda, 0) + '</td><td class="text-right font-bold ' + signClass(dif) + '">' + fmtNum(dif, 0) + '</td></tr>'; }, 3);

      initSimpleDataTable('dtClienteDiff', { pageLength: 8, order: [[3, 'desc']] });
      initSimpleDataTable('dtMaterialDiff', { pageLength: 8, order: [[1, 'desc']] });
      initSimpleDataTable('dtPedidoDiff', { pageLength: 10, order: [[4, 'desc']] });
    })
    .catch(function (err) { console.error('Erro ao carregar KPIs:', err); });

  const filtroCliente = document.getElementById('filtroCliente');
  const filtroMaterial = document.getElementById('filtroMaterial');
  const filtroOp = document.getElementById('filtroOp');
  const filtroDiferenca = document.getElementById('filtroDiferenca');

  if (window.DataTable && DataTable.ext && DataTable.ext.search) {
    DataTable.ext.search.push(function (settings, data) {
      if (!settings || settings.nTable.id !== 'dtPlanilha') return true;
      const difTxt = String(data[4] || '0').replace(/\./g, '').replace(',', '.');
      const dif = Number(difTxt);
      if (planilhaDiffFilter === 'positivo') return dif > 0;
      if (planilhaDiffFilter === 'negativo') return dif < 0;
      if (planilhaDiffFilter === 'zero') return dif === 0;
      return true;
    });
  }

  function aplicarFiltrosTabela() {
    if (!dtPlanilha) return;
    dtPlanilha.column(1).search((filtroCliente.value || '').trim());
    dtPlanilha.column(8).search((filtroMaterial.value || '').trim());
    dtPlanilha.column(0).search((filtroOp.value || '').trim());
    planilhaDiffFilter = filtroDiferenca.value || '';
    dtPlanilha.draw();
  }

  if (window.DataTable) {
    dtPlanilha = new DataTable('#dtPlanilha', {
      pageLength: 25,
      lengthMenu: [10, 25, 50, 100],
      order: [[13, 'desc']],
      scrollX: true,
      language: dtLanguage()
    });
  }

  [filtroCliente, filtroMaterial, filtroOp, filtroDiferenca].forEach(function (el) {
    el.addEventListener('input', aplicarFiltrosTabela);
    el.addEventListener('change', aplicarFiltrosTabela);
  });
})();
</script>

<?php require __DIR__ . '/_layout_bottom.php'; ?>
